<?php
// app/Models/TransferReport.php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class TransferReport extends Model
{
    use HasFactory;

    protected $fillable = [
        'item_id',
        'from_cabin_id',
        'to_cabin_id',
        'quantity',
        'status',
        'initiated_by',
        'notes',
        'transfer_date',
        'completed_at'
    ];

    protected $casts = [
        'transfer_date' => 'datetime',
        'completed_at' => 'datetime',
        'quantity' => 'integer',
    ];

    /**
     * Get the item that was transferred
     */
    public function item(): BelongsTo
    {
        return $this->belongsTo(Item::class);
    }

    /**
     * Get the source cabin
     */
    public function fromCabin(): BelongsTo
    {
        return $this->belongsTo(cabinet::class, 'from_cabin_id');
    }

    public function items(): BelongsTo
    {
        return $this->belongsTo(Item::class, 'item_id');
    }

    /**
     * Get the destination cabin
     */
    public function toCabin(): BelongsTo
    {
        return $this->belongsTo(cabinet::class, 'to_cabin_id');
    }

    /**
     * Get the user who initiated the transfer
     */
    public function initiator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'initiated_by');
    }

    /**
     * Scope a query to filter by date range
     */
    public function scopeDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('transfer_date', [$startDate, $endDate]);
    }

    /**
     * Scope a query to filter by status
     */
    public function scopeStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope a query to filter by cabin
     */
    public function scopeCabinFilter($query, $fromCabin = null, $toCabin = null)
    {
        if ($fromCabin) {
            $query->where('from_cabin_id', $fromCabin);
        }
        
        if ($toCabin) {
            $query->where('to_cabin_id', $toCabin);
        }
        
        return $query;
    }

    /**
     * Scope a query to filter by item
     */
    public function scopeItemFilter($query, $itemId)
    {
        return $query->where('item_id', $itemId);
    }

    /**
     * Scope a query to filter by initiator
     */
    public function scopeInitiator($query, $userId)
    {
        return $query->where('initiated_by', $userId);
    }
}