<?php
// app/Http/Controllers/TransferReportController.php

namespace App\Http\Controllers;

use App\Models\TransferReport;
use App\Models\Cabin;
use App\Models\Item;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use PDF;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\TransferReportsExport;

class TransferReportController extends Controller
{
    /**
     * Display a listing of the transfer reports with filters
     */
    public function index(Request $request): JsonResponse
{
    try {
       $query = TransferReport::with([
    'item:id,name,type_id',
    'item.type:id,picture',
    'fromCabin:id,name',
    'toCabin:id,name',
    'initiator:id,name'
]);


        // Apply filters
        if ($request->has('dateRange') && $request->dateRange !== 'custom') {
            $days = (int) $request->dateRange;
            $startDate = Carbon::now()->subDays($days);
            $endDate = Carbon::now();
            $query->dateRange($startDate, $endDate);
        } elseif ($request->has('startDate') && $request->has('endDate')) {
            $startDate = Carbon::parse($request->startDate);
            $endDate = Carbon::parse($request->endDate)->endOfDay();
            $query->dateRange($startDate, $endDate);
        }

        if ($request->has('cabinFrom') && $request->cabinFrom) {
            $query->where('from_cabin_id', $request->cabinFrom);
        }

        if ($request->has('cabinTo') && $request->cabinTo) {
            $query->where('to_cabin_id', $request->cabinTo);
        }

        if ($request->has('item') && $request->item) {
            $query->where('item_id', $request->item);
        }

        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        if ($request->has('initiatedBy') && $request->initiatedBy) {
            $query->where('initiated_by', $request->initiatedBy);
        }

        // Get pagination parameters
        $perPage = $request->get('perPage', 10);
        $page = $request->get('page', 1);

        // Create a fresh query for chart data without ordering
        $chartQuery = (clone $query)->withoutGlobalScopes()->reorder();
        
        // Execute query with pagination (add ordering here)
        $reports = $query->orderBy('transfer_date', 'desc')
                        ->paginate($perPage, ['*'], 'page', $page);

        // Calculate summary statistics using the chart query (no ordering)
        $summary = $this->calculateSummary($chartQuery);

        // Get chart data using the chart query (no ordering)
        $chartData = $this->getChartData($chartQuery);

        return response()->json([
            'success' => true,
            'data' => $reports->items(),
            'pagination' => [
                'currentPage' => $reports->currentPage(),
                'totalPages' => $reports->lastPage(),
                'totalItems' => $reports->total(),
                'perPage' => $reports->perPage()
            ],
            'summary' => $summary,
            'chartData' => $chartData
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Failed to fetch transfer reports: ' . $e->getMessage()
        ], 500);
    }
}







private function applyFiltersToQuery($query, Request $request)
{
    if ($request->has('dateRange') && $request->dateRange !== 'custom') {
        $days = (int) $request->dateRange;
        $startDate = Carbon::now()->subDays($days);
        $endDate = Carbon::now();
        $query->dateRange($startDate, $endDate);
    } elseif ($request->has('startDate') && $request->has('endDate')) {
        $startDate = Carbon::parse($request->startDate);
        $endDate = Carbon::parse($request->endDate)->endOfDay();
        $query->dateRange($startDate, $endDate);
    }

    if ($request->has('cabinFrom') && $request->cabinFrom) {
        $query->where('from_cabin_id', $request->cabinFrom);
    }

    if ($request->has('cabinTo') && $request->cabinTo) {
        $query->where('to_cabin_id', $request->cabinTo);
    }

    if ($request->has('item') && $request->item) {
        $query->where('item_id', $request->item);
    }

    if ($request->has('status') && $request->status) {
        $query->where('status', $request->status);
    }

    if ($request->has('initiatedBy') && $request->initiatedBy) {
        $query->where('initiated_by', $request->initiatedBy);
    }
    
    return $query;
}

// Then in your index method:
// $chartQuery = TransferReport::query();
// $this->applyFiltersToQuery($chartQuery, $request);


    /**
     * Calculate summary statistics
     */
    private function calculateSummary($query): array
    {
        $total = $query->count();
        $completed = (clone $query)->where('status', 'completed')->count();
        $pending = (clone $query)->where('status', 'pending')->count();
        $failed = (clone $query)->where('status', 'failed')->count();
        $totalItems = (clone $query)->sum('quantity');

        // Calculate average transfer time (in minutes)
        $averageTime = (clone $query)
            ->whereNotNull('completed_at')
            ->where('status', 'completed')
            ->selectRaw('AVG(TIMESTAMPDIFF(MINUTE, transfer_date, completed_at)) as avg_time')
            ->value('avg_time') ?? 0;

        return [
            'totalTransfers' => $total,
            'completed' => $completed,
            'pending' => $pending,
            'failed' => $failed,
            'totalItems' => $totalItems,
            'averageTime' => round($averageTime, 2)
        ];
    }

    /**
     * Get chart data for visualization
     */
    private function getChartData($baseQuery): array
{
    try {
        // Daily transfers data - use fresh query without any ordering
        $dailyQuery = (clone $baseQuery)->reorder();
        $dailyTransfers = $dailyQuery
            ->selectRaw('DATE(transfer_date) as date, COUNT(*) as count')
            ->groupBy(DB::raw('DATE(transfer_date)'))
            ->orderBy('date', 'asc') // Only order by the aliased column
            ->get()
            ->map(function ($item) {
                return [
                    'date' => $item->date,
                    'transfers' => $item->count
                ];
            });

        // Cabin performance data - use fresh query
        $cabinQuery = (clone $baseQuery)->reorder();
        $cabinPerformance = $cabinQuery
            ->join('cabins', 'transfer_reports.from_cabin_id', '=', 'cabins.id')
            ->selectRaw('cabins.name as cabin, COUNT(*) as transfers')
            ->groupBy('transfer_reports.from_cabin_id', 'cabins.name')
            ->orderBy('transfers', 'desc')
            ->get()
            ->map(function ($item) {
                return [
                    'cabin' => $item->cabin,
                    'transfers' => $item->transfers
                ];
            });

        // Item movements data - use fresh query
        $itemQuery = (clone $baseQuery)->reorder();
        $itemMovements = $itemQuery
            ->join('items', 'transfer_reports.item_id', '=', 'items.id')
            ->selectRaw('items.name as item, SUM(quantity) as total_moved')
            ->groupBy('transfer_reports.item_id', 'items.name')
            ->orderByDesc('total_moved')
            ->limit(10)
            ->get()
            ->map(function ($item) {
                return [
                    'item' => $item->item,
                    'total_moved' => (int)$item->total_moved
                ];
            });

        return [
            'dailyTransfers' => $dailyTransfers,
            'cabinPerformance' => $cabinPerformance,
            'itemMovements' => $itemMovements
        ];

    } catch (\Exception $e) {
        // Log error and return empty arrays
        \Log::error('Failed to get chart data: ' . $e->getMessage());
        return [
            'dailyTransfers' => [],
            'cabinPerformance' => [],
            'itemMovements' => []
        ];
    }
}

    /**
     * Export transfer reports
     */
    public function export(Request $request)
    {
        try {
            $format = $request->get('format', 'csv');
            
            if ($format === 'pdf') {
                $reports = $this->getReportsForExport($request);
                $pdf = PDF::loadView('exports.transfer-reports', ['reports' => $reports]);
                return $pdf->download('transfer_reports_' . now()->format('Y-m-d') . '.pdf');
            } else {
                return Excel::download(new TransferReportsExport($request), 'transfer_reports_' . now()->format('Y-m-d') . '.xlsx');
            }

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Export failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get reports data for export
     */
    private function getReportsForExport(Request $request)
    {
        $query = TransferReport::with(['item', 'fromCabin', 'toCabin', 'initiator']);

        // Apply the same filters as index method
        // ... (copy filter logic from index method)

        return $query->orderBy('transfer_date', 'desc')->get();
    }

    /**
     * Get form data for filters (cabins, items, users)
     */
    public function getFormData(): JsonResponse
    {
        try {
            $cabins = Cabin::select('id', 'name')->get();
            $items = Item::select('id', 'name', 'type_id')
            ->with('type:id,picture') // فقط فیلد picture رو از type بگیره
            ->get();

            $users = User::select('id', 'name')->get();

            return response()->json([
                'success' => true,
                'data' => [
                    'cabins' => $cabins,
                    'items' => $items,
                    'users' => $users
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch form data: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created transfer report
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'item_id' => 'required|exists:items,id',
            'from_cabin_id' => 'required|exists:cabins,id',
            'to_cabin_id' => 'required|exists:cabins,id|different:from_cabin_id',
            'quantity' => 'required|integer|min:1',
            'status' => 'required|in:completed,pending,failed',
            'notes' => 'nullable|string|max:500',
            'transfer_date' => 'nullable|date'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();

            $report = TransferReport::create([
                'item_id' => $request->item_id,
                'from_cabin_id' => $request->from_cabin_id,
                'to_cabin_id' => $request->to_cabin_id,
                'quantity' => $request->quantity,
                'status' => $request->status,
                'initiated_by' => auth()->id(),
                'notes' => $request->notes,
                'transfer_date' => $request->transfer_date ?? now(),
                'completed_at' => $request->status === 'completed' ? now() : null
            ]);

            // Load relationships for response
            $report->load(['item', 'fromCabin', 'toCabin', 'initiator']);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Transfer report created successfully',
                'data' => $report
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to create transfer report: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified transfer report
     */
    public function show($id): JsonResponse
    {
        try {
            $report = TransferReport::with(['item', 'fromCabin', 'toCabin', 'initiator'])
                                  ->findOrFail($id);

            return response()->json([
                'success' => true,
                'data' => $report
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Transfer report not found'
            ], 404);
        }
    }

    /**
     * Update the specified transfer report
     */
    public function update(Request $request, $id): JsonResponse
    {
        // Similar to store method with validation and update logic
        // Implementation depends on your requirements
    }

    /**
     * Remove the specified transfer report
     */
    public function destroy($id): JsonResponse
    {
        try {
            $report = TransferReport::findOrFail($id);
            $report->delete();

            return response()->json([
                'success' => true,
                'message' => 'Transfer report deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete transfer report'
            ], 500);
        }
    }
}