<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Http\Resources\BelanceResource;
use App\Http\Resources\MoneyResource;
use App\Models\{
    Item, Purchase, Sidebar, User, Sell, Moneys, Belances, 
    Report, Accounts, Bill
};
use App\Models\CabinStock;
use Illuminate\Http\Request;
use Exception;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class SellsController extends Controller
{
    // Constants for configuration
    const DEFAULT_PER_PAGE = 10;
    const MAIN_ACCOUNT_ID = 1;
    
    /**
     * List sales with pagination and filtering
     */
     
public function dateChange($date)
{
    if (!$date) {
        return null;
    }

    try {
        // Convert ISO format (2025-09-30T19:30:00.000Z) to Y-m-d H:i:s
        return Carbon::parse($date)->format('Y-m-d H:i:s');
    } catch (\Exception $e) {
        throw new \InvalidArgumentException('تاریخ نامعتبر است', 422);
    }
}

public function index(Request $request)
{
    try {
        $isDelete = $request->query('delete', 0);
        $perPage = $request->query('perPage', 10);
        $search = $request->query('search');
        $startDate = $request->input('StartDate');
        $endDate = $request->input('EndDate');
        $user_id = $request->input('user_id');
        
        $query = Sell::with([
                'bill.money', 
                'user', 
                'bill.accounts.account', 
                'stock', 
                'cobin'
            ])
            ->whereHas('bill', function ($query) use ($isDelete) {
                $query->where('isdelete', $isDelete);
            });

        // User filter
        if ($user_id && $user_id != '0') {
            $query->where('user_id', $user_id);
        }
        
        // Date filter - Convert ISO to database format
        if ($startDate && $endDate) {
            $convertedStartDate = $this->dateChange($startDate);
            $convertedEndDate = $this->dateChange($endDate);
            $query->whereBetween('dateInsert', [$convertedStartDate, $convertedEndDate]);
        } else {
            if ($startDate) {
                $convertedStartDate = $this->dateChange($startDate);
                $query->where('dateInsert', '>=', $convertedStartDate);
            }
            
            if ($endDate) {
                $convertedEndDate = $this->dateChange($endDate);
                $query->where('dateInsert', '<=', $convertedEndDate);
            }
        }

        // Search filter
        if ($search && $search !== 'false') {
            $query->where(function($q) use ($search) {
                $q->whereHas('bill', function ($billQuery) use ($search) { 
                    $billQuery->where('temp_customer', 'like', "%$search%")
                          ->orWhereHas('accounts.account', function ($accountQuery) use ($search) {
                              $accountQuery->where('name', 'like', "%$search%");
                          });
                });
            });
        }

        $query->orderBy('dateInsert', 'desc');
        
        // Get paginated results
        $sells = $query->paginate($perPage);
        
        // Process and group data
        $processedData = $sells->getCollection()
            ->groupBy('bill.id')
            ->map(function ($group) {
                $firstItem = $group->first();
                $totalQty = $group->sum('qty') ?? 0;

                return [
                    'bill' => $firstItem->bill,
                    'money' => $firstItem->bill->money,
                    'sells' => $group->values()->toArray(),
                    'totalQty' => $totalQty,
                ];
            })
            ->values();

        // Replace collection with processed data
        $sells->setCollection($processedData);
        
        return response()->json([
            'data' => $sells->items(),
            'total' => $sells->total(),
            'current_page' => $sells->currentPage(),
            'per_page' => $sells->perPage(),
            'last_page' => $sells->lastPage(),
        ], 200);

    } catch (\InvalidArgumentException $e) {
        return response()->json(['error' => $e->getMessage()], 422);
    } catch (\Exception $e) {
        \Log::error('Sell index error: ' . $e->getMessage());
        return response()->json(['error' => 'خطا در دریافت اطلاعات'], 500);
    }
}

    /**
     * Store a new sale
     */
    public function store(Request $request)
    {
        $params = $request->query();
        $validatedData = $this->validateSaleData($request);

        return DB::transaction(function () use ($params, $validatedData) {
            if ($params['Exesting'] === "ok") {
                return $this->handleExistingAccountSale($params, $validatedData);
            } else {
                return $this->handleNewCustomerSale($params, $validatedData);
            }
        });
    }

    /**
     * Show sale details
     */
    public function show($billId)
    {
        $sells = Sell::with(['bill.money', 'user', 'bill.accounts.account', 'stock'])
            ->whereHas('bill', fn($q) => $q->where('id', $billId))
            ->orderBy('dateInsert', 'desc')
            ->get();

        if ($sells->isEmpty()) {
            return response()->json(['message' => 'No data found for this bill.'], 404);
        }

        $firstSell = $sells->first();
        $responseData = [
            'bill' => $firstSell->bill,
            'money' => $firstSell->bill->money,
            'sells' => $sells->toArray(),
        ];

        return response()->json($responseData);
    }

    /**
     * Update a sale
     */
    public function update($id, Request $request)
    {
        $type = $request->query('type');
        $params = $request->query();

        if ($type === 'delete') {
            return $this->softDeleteSale($request,$id);
        }

        $validatedData = $this->validateSaleData($request);
        
        $purchasesToDelete = Sell::where('bill_id', $validatedData['arr'][0]['bill_id'])->get();

        foreach ($purchasesToDelete as $purchase) {
        // اینجا مستقیم مدل رو می‌فرستی
        $this->adjustItemStock($purchase->toArray(), 'increase', $params['prevMoney']);

        // حذف رکورد
        $purchase->delete();
    }


        return DB::transaction(function () use ($params, $validatedData) {
            $this->revertPreviousSaleEffects($params);

            if ($params['Exesting'] === 'ok') {
                return $this->handleExistingAccountUpdate($params, $validatedData);
            } else {
                return $this->handleNewCustomerUpdate($params, $validatedData);
            }
        });
    }

    /**
     * Delete a sale
     */
    public function destroy($id)
    {
        try {
            $sell = Sell::findOrFail($id);
            $sell->delete();
            
            return response()->json([
                'success' => true, 
                'message' => 'Sell was successfully deleted.'
            ], 200);
        } catch (Exception $exception) {
            return response()->json([
                'success' => false, 
                'message' => 'Error occurred while deleting the sell.'
            ], 500);
        }
    }

    // ============ PRIVATE HELPER METHODS ============

    /**
     * Validate sale data
     */
    private function validateSaleData(Request $request): array
    {
        return $request->validate([
            'arr' => 'required|array',
            'arr.*.item_id' => 'required',
            'arr.*.bill_id' => 'sometimes',
            'arr.*.accounts_id' => 'sometimes',
            'arr.*.discount' => 'sometimes',
            'arr.*.qty' => 'required|numeric|min:-2147483648|max:2147483647',
            'arr.*.weight' => 'nullable',
            'arr.*.dateInsert' => 'required',
            'arr.*.rate' => 'nullable|numeric|min:-2147483648|max:2147483647',
            'arr.*.user_id' => 'required',
            'arr.*.isdelete' => 'nullable|boolean',
            'arr.*.purchase_price' => 'nullable|numeric|min:-2147483648|max:2147483647',
            'arr.*.sell_price' => 'nullable|numeric|min:-2147483648|max:2147483647',
            'arr.*.expiry_date' => 'sometimes',
            'arr.*.description' => 'nullable',
            'arr.*.money_id' => 'nullable',
            'arr.*.prevMoney' => 'nullable',
            'arr.*.cobin_id' => 'nullable',
            'arr.*.cabin_stock_id' => 'nullable',
            'arr.*.money_id_purchase' => 'nullable',
        ]);
    }

    /**
     * Handle sale with existing account
     */
    private function handleExistingAccountSale(array $params, array $validatedData)
    {
        $moneyId = $params['money'];
        $accountId = $params['accounts_id'];
        $totalAmount = $params['TotalAmount'];
        $paidAmount = $params['PaidAmount'];
        $DiscountAmount = $params['DiscountAmount'];
        $sellMoney = $moneyId;
        $DateInsert=$params['DateInsert'];
        $date = $this->dateChange($DateInsert);
        $accountBalance = Belances::firstOrCreate(
            ['type_id' => $moneyId, 'account_id' => $accountId],
            [
                'user_id' => $validatedData['arr'][0]['user_id'],
                'belance' => 0,
            ]
        );

        $accountBalance->belance -= ($totalAmount - $paidAmount);
        $accountBalance->save();

        $mainBalance = Belances::firstOrNew([
            'account_id' => self::MAIN_ACCOUNT_ID,
            'type_id' => $moneyId
        ]);
        $mainBalance->belance += $totalAmount;
        $mainBalance->save();

        $bill = Bill::create([
            'accounts_id' => $accountBalance->id,
            'user_id' => $validatedData['arr'][0]['user_id'],
            'total' => $totalAmount,
            'PaidAmount' => $paidAmount,
            'Remain' => $paidAmount - $totalAmount,
            'discountAmount'=>$DiscountAmount,
            'dateInsert' => $date,
            'type' => 'sell',
            'money_id' => $moneyId,
        ]);

        Report::create([
            'isdelete' => 0,
            'user_id' => $validatedData['arr'][0]['user_id'],
            'discription' => 'By Bill Number of ' . $bill->id,
            'amount' => $totalAmount - $paidAmount,
            'date_created' => $date,
            'type' => 'withdraw',
            'account_id' => $accountBalance->id,
            'cach' => $accountBalance->belance,
            'bill_id' => $bill->id,
        ]);

        $money = Moneys::find($moneyId);
        $money->cach += $paidAmount;
        $money->save();

        $this->createSaleItems($validatedData['arr'], $bill->id, $sellMoney);

        return response()->json([
            'message' => 'Report created successfully',
            'bill' => $bill,
            'belance' => new BelanceResource($accountBalance),
            'mainbelance' => new BelanceResource($mainBalance),
            'moneys' => new MoneyResource($money)
        ], 201);
    }

    /**
     * Handle sale with new customer
     */
    private function handleNewCustomerSale(array $params, array $validatedData)
    {
        $moneyId = $params['money'];
        $totalAmount = $params['TotalAmount'];
        $customerName = $params['CustomerName'];
        $DiscountAmount = $params['DiscountAmount'];
        $sellMoney = $moneyId;
        $DateInsert=$params['DateInsert'];
        $date = $this->dateChange($DateInsert);
        $mainBalance = Belances::firstOrNew([
            'account_id' => self::MAIN_ACCOUNT_ID,
            'type_id' => $moneyId
        ]);
        $mainBalance->belance += $totalAmount;
        $mainBalance->save();

        $bill = Bill::create([
            'user_id' => $validatedData['arr'][0]['user_id'],
            'total' => $totalAmount,
            'PaidAmount' => $totalAmount,
            'discountAmount' => $DiscountAmount,
            'Remain' => 0,
            'dateInsert' => $date,
            'type' => 'sell',
            'temp_customer' => $customerName,
            'money_id' => $moneyId,
        ]);

        $money = Moneys::find($moneyId);
        $money->cach += $totalAmount;
        $money->save();

        $this->createSaleItems($validatedData['arr'], $bill->id, $sellMoney);

        return response()->json([
            'message' => 'Report created successfully',
            'bill' => $bill,
            'mainbelance' => new BelanceResource($mainBalance),
            'moneys' => new MoneyResource($money)
        ], 201);
    }

    /**
     * Soft delete a sale
     */
    private function softDeleteSale(Request $request,$id)
    {
        $data = $request->validate([
            'money' => '',
            'bill' => '',
        ]);

        return DB::transaction(function () use ($id) {
            $sells = Sell::where('bill_id', $id)->get();
            
            foreach ($sells as $sell) {
                $this->revertItemStock($sell);
                $sell->isdelete = 1;
                $sell->save();
            }

            $bill = Bill::find($id);
            $bill->isdelete = 1;
            $bill->save();

            $money = Moneys::find($bill->money_id);
            $money->cach -= $bill->PaidAmount;
            $money->save();

            $mainBalance = Belances::where('type_id', $money->id)
                ->where('account_id', self::MAIN_ACCOUNT_ID)
                ->first();
            $mainBalance->belance -= $bill->total;
            $mainBalance->save();

            $c_belance=Belances::where('id',$bill->accounts_id)->first();
            if($c_belance){
                $c_belance->belance+=$bill->total;
                $c_belance->belance-= $bill->PaidAmount;
                $c_belance->save();
            }

            Report::where('bill_id', $bill->id)->update(['isdelete' => 1]);

            return response()->json(['success' => true]);
        });
    }

    /**
     * Revert effects of previous sale before update
     */
    private function revertPreviousSaleEffects(array $params)
    {
        if ($params['Accounts_id']) {
            $prevBalance = Belances::find($params['Accounts_id']);
            $prevMainBalance = Belances::where('account_id', self::MAIN_ACCOUNT_ID)
                ->where('type_id', $prevBalance->type_id)
                ->first();
            
            $prevMainBalance->belance -= $params['primaryTotalAmount'];
            $prevMainBalance->save();
            
            $prevBalance->belance += $params['primaryTotalAmount'] - $params['primaryPaidAmount'];
            $prevBalance->save();
            
            $prevMoney = Moneys::find($params['prevMoney']);
            $prevMoney->cach -= $params['primaryPaidAmount'];
            $prevMoney->save();
            
            Report::where('bill_id', $params['Bill_id'])->delete();
        } else {
            $prevMainBalance = Belances::where('account_id', self::MAIN_ACCOUNT_ID)
                ->where('type_id', $params['prevMoney'])
                ->first();
            $prevMainBalance->belance -= $params['primaryTotalAmount'];
            $prevMainBalance->save();
            
            $prevMoney = Moneys::find($params['prevMoney']);
            $prevMoney->cach -= (int)$params['primaryPaidAmount'];
            $prevMoney->save();
        }
    }

    /**
     * Handle update with existing account
     */
    private function handleExistingAccountUpdate(array $params, array $validatedData)
    {
        $moneyId = $params['money'];
        $accountId = $params['accounts_id'];
        $totalAmount = $params['TotalAmount'];
        $paidAmount = $params['PaidAmount'];
        $billId = $params['Bill_id'];
        $DiscountAmount = $params['DiscountAmount'];
        $DateInsert=$params['DateInsert'];
        $date = $this->dateChange($DateInsert);
        $sellMoney = $moneyId;

        $accountBalance = Belances::firstOrCreate(
            ['type_id' => $moneyId, 'account_id' => $accountId],
            [
                'user_id' => $validatedData['arr'][0]['user_id'],
                'belance' => 0,
            ]
        );

        $accountBalance->belance -= ($totalAmount - $paidAmount);
        $accountBalance->save();

        $mainBalance = Belances::where('account_id', self::MAIN_ACCOUNT_ID)
            ->where('type_id', $moneyId)
            ->first();
        $mainBalance->belance += $totalAmount;
        $mainBalance->save();

        $bill = Bill::find($billId);
        $bill->update([
            'accounts_id' => $accountBalance->id,
            'user_id' => $validatedData['arr'][0]['user_id'],
            'total' => $totalAmount,
            'PaidAmount' => $paidAmount,
            'Remain' => $paidAmount - $totalAmount,
            'dateInsert' => $date,
            'discountAmount'=>$DiscountAmount,
            'money_id' => $moneyId,
            'temp_customer' => null,
        ]);

        $money = Moneys::find($moneyId);
        $money->cach += $paidAmount;
        $money->save();

        Report::updateOrCreate(
            ['bill_id' => $billId],
            [
                'isdelete' => 0,
                'user_id' => $validatedData['arr'][0]['user_id'],
                'discription' => 'By Bill Number of ' . $billId,
                'amount' => $totalAmount - $paidAmount,
                'date_created' => $date,
                'type' => 'withdraw',
                'account_id' => $accountBalance->id,
            ]
        );

        $this->processSaleItemsUpdate($validatedData['arr'], $billId, $sellMoney);
        $bill->load('accounts.account','money');
        return response()->json([
            'message' => 'Report updated successfully',
            'bill' => $bill,
            'belance' => new BelanceResource($accountBalance),
            'mainbelance' => new BelanceResource($mainBalance),
            'moneys' => new MoneyResource($money)
        ], 200);
    }

    /**
     * Handle update with new customer
     */
    private function handleNewCustomerUpdate(array $params, array $validatedData)
    {
        $moneyId = $params['money'];
        $totalAmount = $params['TotalAmount'];
        $paidAmount = $params['PaidAmount'];
        $billId = $params['Bill_id'];
        $customerName = $params['CustomerName'];
        $sellMoney = $moneyId;
        $DiscountAmount = $params['DiscountAmount'];
        $DateInsert=$params['DateInsert'];
        $date = $this->dateChange($DateInsert);
        $mainBalance = Belances::where('account_id', self::MAIN_ACCOUNT_ID)
            ->where('type_id', $moneyId)
            ->first();
        $mainBalance->belance += $totalAmount;
        $mainBalance->save();

        $money = Moneys::find($moneyId);
        $money->cach += $totalAmount;
        $money->save();

        $bill = Bill::find($billId);
        $bill->update([
            'total' => $totalAmount,
            'PaidAmount' => $paidAmount,
            'Remain' => 0,
            'temp_customer' => $customerName,
            'discountAmount'=>$DiscountAmount,
            'money_id' => $moneyId,
            'accounts_id' => null,
            'dateInsert' => $date,
        ]);

        $this->processSaleItemsUpdate($validatedData['arr'], $billId, $sellMoney);
        $bill->load('accounts.account','money');
        return response()->json([
            'message' => 'Report updated successfully',
            'bill' => $bill,
            'mainbelance' => new BelanceResource($mainBalance),
            'moneys' => new MoneyResource($money)
        ], 200);
    }

    /**
     * Create sale items
     */
    private function createSaleItems(array $items, int $billId, $moneyId)
    {
        foreach ($items as $item) {
            $item['purchase_price'] = $this->adjustItemStock($item, 'decrease',$item['money_id_purchase']);
            
            $item['bill_id'] = $billId;
            $item['money_id'] = $moneyId;
            $item['dateInsert'] = $this->dateChange($item['dateInsert']);
            $item['expiry_date'] = $this->dateChange($item['expiry_date']);
            // $purchase = Purchase::when(isset($item['cabin_stock_id']), function ($query) use ($item) {
            //     return $query->where('id', $item['cabin_stock_id']);
            //     })->first();

            //     if ($purchase) {
            //         $purchase->update(['insell' => 1]);
            //         // $item['purchase_price'] = $purchase->purchase_price;
            //     }
            Sell::create($item);
        }
    }

    /**
     * Process sale items update (create new or revert old)
     */
    private function processSaleItemsUpdate(array $items, int $billId, $moneyId)
    {
        foreach ($items as $item) {
            // if (empty($item['bill_id'])) {
                $this->adjustItemStock($item, 'decrease',$item['money_id_purchase']);
                
                $item['bill_id'] = $billId;
                $item['money_id'] = $moneyId;
            
                // $item['dateInsert'] = $this->formatDate($item['dateInsert']);
                // $item['expiry_date'] = isset($item['expiry_date']) ? $this->formatDate($item['expiry_date']) : null;
                
            $item['dateInsert'] = $this->dateChange($item['dateInsert']);
            // $purchase = Purchase::when(isset($item['purchase_id']), function ($query) use ($item) {
            //     return $query->where('id', $item['purchase_id']);
            // })->first();

            // if ($purchase) {
            //     $purchase->update(['insell' => 1]);
            //     // $item['purchase_price'] = $purchase->purchase_price;
            // }


            $item['expiry_date'] = $this->dateChange($item['expiry_date']);
                Sell::create($item);
            }
        // }
    }

    /**
     * Adjust item stock (increase or decrease)
     */
    private function adjustItemStock( $item,  $action, $money)
    {

        
        $product = Item::find($item['item_id']);
        
        
        
        $multiplier = ($action === 'increase') ? 1 : -1;
        $purchasePrice=0;
        if($multiplier===-1){
            if($item['cabin_stock_id']==="none"){
                $cabinStock = CabinStock::where('id',$item['cabin_stock_id'])->get();
            }else{
                $cabinStock = CabinStock::where('purchase_price',$item['purchase_price'])->where('cobin_id','1')->where('item_id',$item['item_id'])->where('money_id','1')->get();
            }
                if ($cabinStock->isNotEmpty()) {
                    $firstStock = $cabinStock->first();
                    if ($firstStock->qty - $item['qty'] >= 0) {
                        // مقدار کافی هست
                        $firstStock->qty -= $item['qty'];
                        $firstStock->save();
                    } else {
                        // مقدار کافی نیست → باید از چند رکورد کم بشه
                        $qty =(int) $item['qty'];
                        $i = 0;

                        while ($qty > 0 ) {
                            if ($cabinStock[$i]->qty >= $qty) {
                                // اگر موجودی کافی داشت
                                $cabinStock[$i]->qty -= $qty;
                                $cabinStock[$i]->save();
                                break;
                            }elseif(!isset($cabinStock[$i+1])){
                                $cabinStock[$i]->qty -=$qty;
                                $cabinStock[$i]->save();
                                break;
                            }
                            else {
                                // اگر موجودی کافی نبود، کل موجودی این cabin رو مصرف می‌کنیم
                                $main=$qty;
                                $qty -= $cabinStock[$i]->qty;
                                $cabinStock[$i]->qty -= $main;
                                $cabinStock[$i]->save();
                                if($qty===0){
                                    break;
                                }
                            }
                            $i++;
                        }
                    }
                    $purchasePrice=$firstStock->purchase_price;
                }          
        }else{
            if($item['cabin_stock_id']==="none"){
                $cabinStock = CabinStock::where('id',$item['cabin_stock_id'])->get();
            }else{
                $cabinStock = CabinStock::where('purchase_price',$item['purchase_price'])->where('cobin_id','1')->where('item_id',$item['item_id'])->where('money_id','1')->get();
            }
                if ($cabinStock->isNotEmpty()) {
                    $firstStock = $cabinStock->first();
                        // مقدار کافی هست
                        $firstStock->qty += $item['qty'];
                        $firstStock->save();
                        $purchasePrice=$firstStock->purchase_price;
                }
        }

        $product->qty += $multiplier * $item['qty'];
        $product->sell_price = $item['sell_price'];
        
        if (Sidebar::first()->type === 'gold') {
            $product->weight = ($action === 'increase') 
                ? bcadd($product->weight, $item['weight'], 3)
                : bcsub($product->weight, $item['weight'], 3);
        }
        
        $product->save();
        return $purchasePrice;
        // if (isset($item['purchase_id'])) {
        //     $purchase = Purchase::find($item['purchase_id']);
        //     $purchase->remainqty += $multiplier * $item['qty'];
        //     $purchase->save();
        // }
    }

    /**
     * Revert item stock (alias for adjustItemStock with increase)
     */
    private function revertItemStock($sell)
    {
        // $this->adjustItemStock($sell->toArray(), 'increase',$sell->toArray()->prevMoney);
        $sellArray = $sell->toArray();
        $this->adjustItemStock($sellArray, 'increase', $sellArray['money_id']);

    }

    /**
     * Format date string by removing Z and adding timezone
     */
    private function formatDate(?string $date): ?string
    {
        return $date ? rtrim($date, 'Z') . '+00:00' : null;
    }

    /**
     * Paginate a collection manually
     */
    private function paginateCollection($items, $perPage, $page)
    {
        return new \Illuminate\Pagination\LengthAwarePaginator(
            $items->forPage($page, $perPage),
            $items->count(),
            $perPage,
            $page
        );
    }
}