<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Http\Resources\BelanceResource;
use App\Http\Resources\MoneyResource;
use App\Models\{
    Item, Purchase, Sidebar, User, sellBar, Moneys, Belances, 
    Report, Accounts, Bill
};
use Illuminate\Http\Request;
use Exception;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class SellBarController extends Controller
{
    // Constants for configuration
    const DEFAULT_PER_PAGE = 10;
    const MAIN_ACCOUNT_ID = 1;
    
    /**
     * List sales with pagination and filtering
     */
     
     public function dateChange($date)
{
    if ($date) {
        try {
            // تبدیل تاریخ ISO8601 به فرمت قابل قبول MySQL
            return Carbon::parse($date)->format('Y-m-d H:i:s');
        } catch (\Exception $e) {
            return response()->json(['error' => 'تاریخ نامعتبر است'], 422);
        }
    }

    return null; // اگر مقدار نداشت
}
public function index(Request $request)
{
    $isDelete = $request->query('delete');
    $perPage = $request->query('perPage', 10);
    $search = $request->query('search');
    $startDate = $request->input('StartDate', false);
    $endDate = $request->input('EndDate', false);
    $user_id = $request->input('user_id', false);
    
    $query = sellBar::with(['bill.money', 'user', 'bill.accounts.account', 'stock', 'cobin'])
        ->whereHas('bill', function ($query) use ($isDelete) {
            $query->where('isdelete', $isDelete);
        });

    if ($user_id && $user_id != '0') {
        $query->where('user_id', $user_id);
    }
    
    if ($startDate && $endDate) {
        $query->whereBetween('dateInsert', [$startDate, $endDate]);
    }
    
    if ($startDate) {
        $query->where('dateInsert', '>=', $startDate);
    }
    
    if ($endDate) {
        $query->where('dateInsert', '<=', $endDate);
    }

    if ($search !== 'false') {
        $query->whereHas('bill', function ($q) use ($search) { 
            $q->where('temp_customer', 'like', "%$search%")
              ->orWhereHas('accounts.account', function ($b) use ($search) {
                  $b->where('name', 'like', "%$search%");
              });
        });
    }

    $query->orderBy('dateInsert', 'desc');
    
    $reports = $query->get()
        ->groupBy(function ($item) {
            return $item->bill->id;
        })
        ->map(function ($group) {
            // محاسبه جمع مقادیر از تمام sellهای مربوط به این bill
            $totalQty = $group->sum(function ($sell) {
                return $sell->qty ?? 0; // یا هر فیلدی که بخوای جمع بزنی
            });

            return [
                'bill' => $group->first()->bill,
                'money' => $group->first()->bill->money,
                'sells' => $group->toArray(),
                'totalQty' => $totalQty, // جمع کل qty
            ];
        })
        ->values();

    $currentPage = $request->query('page', 1);
    $paginatedData = $reports->slice(($currentPage - 1) * $perPage, $perPage)->toArray();
    
    return response()->json([
        'data' => $paginatedData,
        'total' => $reports->count(),
        'current_page' => $currentPage,
        'per_page' => $perPage,
        'last_page' => ceil($reports->count() / $perPage),
    ], 200);
}


    /**
     * Store a new sale
     */
    public function store(Request $request)
    {
        $params = $request->query();
        $validatedData = $this->validateSaleData($request);

        return DB::transaction(function () use ($params, $validatedData) {
            if ($params['Exesting'] === "ok") {
                return $this->handleExistingAccountSale($params, $validatedData);
            } else {
                return $this->handleNewCustomerSale($params, $validatedData);
            }
        });
    }

    /**
     * Show sale details
     */
    public function show($billId)
    {
        $sells = sellBar::with(['bill.money', 'user', 'bill.accounts.account', 'stock'])
            ->whereHas('bill', fn($q) => $q->where('id', $billId))
            ->orderBy('dateInsert', 'desc')
            ->get();

        if ($sells->isEmpty()) {
            return response()->json(['message' => 'No data found for this bill.'], 404);
        }

        $firstSell = $sells->first();
        $responseData = [
            'bill' => $firstSell->bill,
            'money' => $firstSell->bill->money,
            'sells' => $sells->toArray(),
        ];

        return response()->json($responseData);
    }

    /**
     * Update a sale
     */
    public function update($id, Request $request)
    {
        $type = $request->query('type');
        $params = $request->query();

        if ($type === 'delete') {
            return $this->softDeleteSale($request);
        }

        $validatedData = $this->validateSaleData($request);
        
        return DB::transaction(function () use ($params, $validatedData) {
            $this->revertPreviousSaleEffects($params);

            if ($params['Exesting'] === 'ok') {
                return $this->handleExistingAccountUpdate($params, $validatedData);
            } else {
                return $this->handleNewCustomerUpdate($params, $validatedData);
            }
        });
    }

    /**
     * Delete a sale
     */
    public function destroy($id)
    {
        try {
            $sell = sellBar::findOrFail($id);
            $sell->delete();
            
            return response()->json([
                'success' => true, 
                'message' => 'sellBar was successfully deleted.'
            ], 200);
        } catch (Exception $exception) {
            return response()->json([
                'success' => false, 
                'message' => 'Error occurred while deleting the sell.'
            ], 500);
        }
    }

    // ============ PRIVATE HELPER METHODS ============

    /**
     * Validate sale data
     */
    private function validateSaleData(Request $request): array
    {
        return $request->validate([
            'arr' => 'required|array',
            'arr.*.item_id' => 'required',
            'arr.*.bill_id' => 'sometimes',
            'arr.*.accounts_id' => 'sometimes',
            'arr.*.qty' => 'required|numeric|min:-2147483648|max:2147483647',
            'arr.*.weight' => 'nullable',
            'arr.*.dateInsert' => 'required',
            'arr.*.rate' => 'nullable|numeric|min:-2147483648|max:2147483647',
            'arr.*.user_id' => 'required',
            'arr.*.isdelete' => 'nullable|boolean',
            'arr.*.purchase_price' => 'nullable|numeric|min:-2147483648|max:2147483647',
            'arr.*.sell_price' => 'nullable|numeric|min:-2147483648|max:2147483647',
            'arr.*.expiry_date' => 'sometimes',
            'arr.*.description' => 'nullable',
            'arr.*.money_id' => 'nullable',
            'arr.*.cobin_id' => 'nullable',
            'arr.*.purchase_id' => 'nullable',
        ]);
    }

    /**
     * Handle sale with existing account
     */
    private function handleExistingAccountSale(array $params, array $validatedData)
    {
        $moneyId = $params['money'];
        $accountId = $params['accounts_id'];
        $totalAmount = $params['TotalAmount'];
        $paidAmount = $params['PaidAmount'];
        $sellMoney = $moneyId;
        $date = $this->dateChange($validatedData['arr'][0]['dateInsert']);

        $accountBalance = Belances::firstOrCreate(
            ['type_id' => $moneyId, 'account_id' => $accountId],
            [
                'user_id' => $validatedData['arr'][0]['user_id'],
                'belance' => 0,
            ]
        );

        $accountBalance->belance -= ($totalAmount - $paidAmount);
        $accountBalance->save();

        $mainBalance = Belances::firstOrNew([
            'account_id' => self::MAIN_ACCOUNT_ID,
            'type_id' => $moneyId
        ]);
        $mainBalance->belance += $totalAmount;
        $mainBalance->save();

        $bill = Bill::create([
            'accounts_id' => $accountBalance->id,
            'user_id' => $validatedData['arr'][0]['user_id'],
            'total' => $totalAmount,
            'PaidAmount' => $paidAmount,
            'Remain' => $paidAmount - $totalAmount,
            'dateInsert' => $date,
            'type' => 'sell',
            'money_id' => $moneyId,
        ]);

        Report::create([
            'isdelete' => 0,
            'user_id' => $validatedData['arr'][0]['user_id'],
            'discription' => 'By Bill Number of ' . $bill->id,
            'amount' => $totalAmount - $paidAmount,
            'date_created' => $date,
            'type' => 'withdraw',
            'account_id' => $accountBalance->id,
            'cach' => $accountBalance->belance,
            'bill_id' => $bill->id,
        ]);

        $money = Moneys::find($moneyId);
        $money->cach += $paidAmount;
        $money->save();

        $this->createSaleItems($validatedData['arr'], $bill->id, $sellMoney);

        return response()->json([
            'message' => 'Report created successfully',
            'bill' => $bill,
            'belance' => new BelanceResource($accountBalance),
            'mainbelance' => new BelanceResource($mainBalance),
            'moneys' => new MoneyResource($money)
        ], 201);
    }

    /**
     * Handle sale with new customer
     */
    private function handleNewCustomerSale(array $params, array $validatedData)
    {
        $moneyId = $params['money'];
        $totalAmount = $params['TotalAmount'];
        $customerName = $params['CustomerName'];
        $sellMoney = $moneyId;
        $date = $this->dateChange($validatedData['arr'][0]['dateInsert']);

        $mainBalance = Belances::firstOrNew([
            'account_id' => self::MAIN_ACCOUNT_ID,
            'type_id' => $moneyId
        ]);
        $mainBalance->belance += $totalAmount;
        $mainBalance->save();

        $bill = Bill::create([
            'user_id' => $validatedData['arr'][0]['user_id'],
            'total' => $totalAmount,
            'PaidAmount' => $totalAmount,
            'Remain' => 0,
            'dateInsert' => $date,
            'type' => 'sell',
            'temp_customer' => $customerName,
            'money_id' => $moneyId,
        ]);

        $money = Moneys::find($moneyId);
        $money->cach += $totalAmount;
        $money->save();

        $this->createSaleItems($validatedData['arr'], $bill->id, $sellMoney);

        return response()->json([
            'message' => 'Report created successfully',
            'bill' => $bill,
            'mainbelance' => new BelanceResource($mainBalance),
            'moneys' => new MoneyResource($money)
        ], 201);
    }

    /**
     * Soft delete a sale
     */
    private function softDeleteSale(Request $request)
    {
        $data = $request->validate([
            'money' => 'required|numeric',
            'bill' => 'required|numeric',
        ]);

        return DB::transaction(function () use ($data) {
            $sells = sellBar::where('bill_id', $data['bill'])->get();
            
            foreach ($sells as $sell) {
                $this->revertItemStock($sell);
                $sell->isdelete = 1;
                $sell->save();
            }

            $bill = Bill::find($data['bill']);
            $bill->isdelete = 1;
            $bill->save();

            $money = Moneys::find($data['money']);
            $money->cach -= $bill->PaidAmount;
            $money->save();

            $mainBalance = Belances::where('type_id', $money->id)
                ->where('account_id', self::MAIN_ACCOUNT_ID)
                ->first();
            $mainBalance->belance -= $bill->total;
            $mainBalance->save();

            Report::where('bill_id', $bill->id)->update(['isdelete' => 1]);

            return response()->json(['success' => true]);
        });
    }

    /**
     * Revert effects of previous sale before update
     */
    private function revertPreviousSaleEffects(array $params)
    {
        if ($params['Accounts_id']) {
            $prevBalance = Belances::find($params['Accounts_id']);
            $prevMainBalance = Belances::where('account_id', self::MAIN_ACCOUNT_ID)
                ->where('type_id', $prevBalance->type_id)
                ->first();
            
            $prevMainBalance->belance -= $params['primaryTotalAmount'];
            $prevMainBalance->save();
            
            $prevBalance->belance += $params['primaryTotalAmount'] - $params['primaryPaidAmount'];
            $prevBalance->save();
            
            $prevMoney = Moneys::find($params['money']);
            $prevMoney->cach -= $params['primaryPaidAmount'];
            $prevMoney->save();
            
            Report::where('bill_id', $params['Bill_id'])->delete();
        } else {
            $prevMainBalance = Belances::where('account_id', self::MAIN_ACCOUNT_ID)
                ->where('type_id', $params['prevMoney'])
                ->first();
            $prevMainBalance->belance -= $params['primaryTotalAmount'];
            $prevMainBalance->save();
            
            $prevMoney = Moneys::find($params['money']);
            $prevMoney->cach -= $params['primaryPaidAmount'];
            $prevMoney->save();
        }
    }

    /**
     * Handle update with existing account
     */
    private function handleExistingAccountUpdate(array $params, array $validatedData)
    {
        $moneyId = $params['money'];
        $accountId = $params['accounts_id'];
        $totalAmount = $params['TotalAmount'];
        $paidAmount = $params['PaidAmount'];
        $billId = $params['Bill_id'];
        $sellMoney = $moneyId;
        $date = $this->dateChange($validatedData['arr'][0]['dateInsert']);

        $accountBalance = Belances::firstOrCreate(
            ['type_id' => $moneyId, 'account_id' => $accountId],
            [
                'user_id' => $validatedData['arr'][0]['user_id'],
                'belance' => 0,
            ]
        );

        $accountBalance->belance -= ($totalAmount - $paidAmount);
        $accountBalance->save();

        $mainBalance = Belances::where('account_id', self::MAIN_ACCOUNT_ID)
            ->where('type_id', $moneyId)
            ->first();
        $mainBalance->belance += $totalAmount;
        $mainBalance->save();

        $bill = Bill::find($billId);
        $bill->update([
            'accounts_id' => $accountBalance->id,
            'user_id' => $validatedData['arr'][0]['user_id'],
            'total' => $totalAmount,
            'PaidAmount' => $paidAmount,
            'Remain' => $paidAmount - $totalAmount,
            'dateInsert' => $date,
            'money_id' => $moneyId,
            'temp_customer' => null,
        ]);

        $money = Moneys::find($moneyId);
        $money->cach += $paidAmount;
        $money->save();

        Report::updateOrCreate(
            ['bill_id' => $billId],
            [
                'isdelete' => 0,
                'user_id' => $validatedData['arr'][0]['user_id'],
                'discription' => 'By Bill Number of ' . $billId,
                'amount' => $totalAmount - $paidAmount,
                'date_created' => $date,
                'type' => 'withdraw',
                'account_id' => $accountBalance->id,
            ]
        );

        $this->processSaleItemsUpdate($validatedData['arr'], $billId, $sellMoney);

        return response()->json([
            'message' => 'Report updated successfully',
            'bill' => $bill,
            'belance' => new BelanceResource($accountBalance),
            'mainbelance' => new BelanceResource($mainBalance),
            'moneys' => new MoneyResource($money)
        ], 200);
    }

    /**
     * Handle update with new customer
     */
    private function handleNewCustomerUpdate(array $params, array $validatedData)
    {
        $moneyId = $params['money'];
        $totalAmount = $params['TotalAmount'];
        $paidAmount = $params['PaidAmount'];
        $billId = $params['Bill_id'];
        $customerName = $params['CustomerName'];
        $sellMoney = $moneyId;
        $date = $this->dateChange($validatedData['arr'][0]['dateInsert']);

        $mainBalance = Belances::where('account_id', self::MAIN_ACCOUNT_ID)
            ->where('type_id', $moneyId)
            ->first();
        $mainBalance->belance += $totalAmount;
        $mainBalance->save();

        $money = Moneys::find($moneyId);
        $money->cach += $totalAmount;
        $money->save();

        $bill = Bill::find($billId);
        $bill->update([
            'total' => $totalAmount,
            'PaidAmount' => $paidAmount,
            'Remain' => 0,
            'temp_customer' => $customerName,
            'money_id' => $moneyId,
            'accounts_id' => null,
            'dateInsert' => $date,
        ]);

        $this->processSaleItemsUpdate($validatedData['arr'], $billId, $sellMoney);

        return response()->json([
            'message' => 'Report updated successfully',
            'bill' => $bill,
            'mainbelance' => new BelanceResource($mainBalance),
            'moneys' => new MoneyResource($money)
        ], 200);
    }

    /**
     * Create sale items
     */
    private function createSaleItems(array $items, int $billId, $moneyId)
    {
        foreach ($items as $item) {
            $this->adjustItemStock($item, 'decrease');
            
            $item['bill_id'] = $billId;
            $item['money_id'] = $moneyId;
            $item['dateInsert'] = $this->dateChange($item['dateInsert']);
            $item['expiry_date'] = $this->dateChange($item['expiry_date']);
            
            Purchase::when(isset($item['purchase_id']), function ($query) use ($item) {
                $query->where('id', $item['purchase_id'])->update(['insell' => 1]);
            });
            
            sellBar::create($item);
        }
    }

    /**
     * Process sale items update (create new or revert old)
     */
    private function processSaleItemsUpdate(array $items, int $billId, $moneyId)
    {
        foreach ($items as $item) {
            if (empty($item['bill_id'])) {
                $this->adjustItemStock($item, 'decrease');
                
                $item['bill_id'] = $billId;
                $item['money_id'] = $moneyId;
            
                // $item['dateInsert'] = $this->formatDate($item['dateInsert']);
                // $item['expiry_date'] = isset($item['expiry_date']) ? $this->formatDate($item['expiry_date']) : null;
                
            $item['dateInsert'] = $this->dateChange($item['dateInsert']);
            $item['expiry_date'] = $this->dateChange($item['expiry_date']);
                sellBar::create($item);
            } else {
                $this->adjustItemStock($item, 'increase');
            }
        }
    }

    /**
     * Adjust item stock (increase or decrease)
     */
    private function adjustItemStock(array $item, string $action)
    {
        $product = Item::find($item['item_id']);
        $multiplier = ($action === 'increase') ? 1 : -1;
        
        // $product->qty += $multiplier * $item['qty'];
        $product->sell_price = $item['sell_price'];
        
        if (Sidebar::first()->type === 'gold') {
            $product->weight = ($action === 'increase') 
                ? bcadd($product->weight, $item['weight'], 3)
                : bcsub($product->weight, $item['weight'], 3);
        }
        
        $product->save();
        
        if (isset($item['purchase_id'])) {
            $purchase = Purchase::find($item['purchase_id']);
            // $purchase->remainqty += $multiplier * $item['qty'];
            $purchase->save();
        }
    }

    /**
     * Revert item stock (alias for adjustItemStock with increase)
     */
    private function revertItemStock($sell)
    {
        $this->adjustItemStock($sell->toArray(), 'increase');
    }

    /**
     * Format date string by removing Z and adding timezone
     */
    private function formatDate(?string $date): ?string
    {
        return $date ? rtrim($date, 'Z') . '+00:00' : null;
    }

    /**
     * Paginate a collection manually
     */
    private function paginateCollection($items, $perPage, $page)
    {
        return new \Illuminate\Pagination\LengthAwarePaginator(
            $items->forPage($page, $perPage),
            $items->count(),
            $perPage,
            $page
        );
    }
}