<?php

namespace App\Http\Controllers;

use App\Models\Purchase;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class PurchaseReportController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $query = Purchase::with(['user', 'stock', 'money'])
                ->where('isdelete', false);

            // Search filters
            if ($request->has('search') && !empty($request->search)) {
                $searchTerm = $request->search;
                $query->where(function($q) use ($searchTerm) {
                    $q->where('description', 'like', "%{$searchTerm}%")
                      ->orWhereHas('stock', function($q) use ($searchTerm) {
                          $q->where('name', 'like', "%{$searchTerm}%");
                      });
                });
            }

            // Date range filter
            if ($request->has('start_date') && !empty($request->start_date)) {
                $query->where('dateInsert', '>=', $request->start_date);
            }

            if ($request->has('end_date') && !empty($request->end_date)) {
                $query->where('dateInsert', '<=', $request->end_date);
            }

            // User filter
            if ($request->has('user_id') && !empty($request->user_id)) {
                $query->where('user_id', $request->user_id);
            }

            // Status filter (you might need to adjust this based on your business logic)
            if ($request->has('status') && !empty($request->status)) {
                $query->where('status', $request->status);
            }

            // Sorting
            $sortField = $request->get('sort_field', 'dateInsert');
            $sortDirection = $request->get('sort_direction', 'desc');
            
            if (in_array($sortField, ['dateInsert', 'qty', 'purchase_price', 'sell_price'])) {
                $query->orderBy($sortField, $sortDirection);
            }

            // Pagination
            $perPage = $request->get('per_page', 10);
            $Purchases = $query->paginate($perPage);

            // Calculate statistics
            $stats = [
                'total_items' => $query->count(),
                'total_qty' => $query->sum('qty'),
                'total_purchase_value' => $query->sum(DB::raw('qty * purchase_price')),
                'total_sell_value' => $query->sum(DB::raw('qty * sell_price')),
            ];

            return response()->json([
                'data' => $Purchases,
                'stats' => $stats,
                'message' => 'Purchase items retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Error retrieving purchase items: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'qty' => 'required|integer|min:1',
            'weight' => 'nullable|numeric',
            'dateInsert' => 'required|date',
            'rate' => 'nullable|integer',
            'user_id' => 'required|exists:users,id',
            'purchase_price' => 'required|integer|min:0',
            'sell_price' => 'required|integer|min:0',
            'expiry_date' => 'nullable|date',
            'description' => 'nullable|string',
            'stocks_id' => 'nullable|exists:stocks,id',
            'bill_id' => 'nullable|integer',
            'return_sell_id' => 'nullable|integer',
            'money_id' => 'nullable|exists:moneys,id',
            'cobin_id' => 'nullable|integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $Purchase = Purchase::create($request->all());

            return response()->json([
                'data' => $Purchase,
                'message' => 'Purchase item created successfully'
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Error creating purchase item: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        try {
            $Purchase = Purchase::with(['user', 'stock', 'money'])
                ->where('isdelete', false)
                ->findOrFail($id);

            return response()->json([
                'data' => $Purchase,
                'message' => 'Purchase item retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Purchase item not found'
            ], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            $Purchase = Purchase::where('isdelete', false)->findOrFail($id);

            $validator = Validator::make($request->all(), [
                'qty' => 'sometimes|required|integer|min:1',
                'weight' => 'nullable|numeric',
                'dateInsert' => 'sometimes|required|date',
                'rate' => 'nullable|integer',
                'user_id' => 'sometimes|required|exists:users,id',
                'purchase_price' => 'sometimes|required|integer|min:0',
                'sell_price' => 'sometimes|required|integer|min:0',
                'expiry_date' => 'nullable|date',
                'description' => 'nullable|string',
                'stocks_id' => 'nullable|exists:stocks,id',
                'bill_id' => 'nullable|integer',
                'return_sell_id' => 'nullable|integer',
                'money_id' => 'nullable|exists:moneys,id',
                'cobin_id' => 'nullable|integer'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'errors' => $validator->errors()
                ], 422);
            }

            $Purchase->update($request->all());

            return response()->json([
                'data' => $Purchase,
                'message' => 'Purchase item updated successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Error updating purchase item: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        try {
            $Purchase = Purchase::where('isdelete', false)->findOrFail($id);
            
            // Soft delete by setting isdelete to true
            $Purchase->update(['isdelete' => true]);

            return response()->json([
                'message' => 'Purchase item deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Error deleting purchase item: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get purchase statistics for reporting
     */
    public function getStatistics(Request $request)
    {
        try {
            $query = Purchase::where('isdelete', false);

            // Date range filter
            if ($request->has('start_date') && !empty($request->start_date)) {
                $query->where('dateInsert', '>=', $request->start_date);
            }

            if ($request->has('end_date') && !empty($request->end_date)) {
                $query->where('dateInsert', '<=', $request->end_date);
            }

            $stats = [
                'total_items' => $query->count(),
                'total_qty' => $query->sum('qty'),
                'total_purchase_value' => $query->sum(DB::raw('qty * purchase_price')),
                'total_sell_value' => $query->sum(DB::raw('qty * sell_price')),
                'avg_purchase_price' => $query->avg('purchase_price'),
                'avg_sell_price' => $query->avg('sell_price'),
            ];

            return response()->json([
                'data' => $stats,
                'message' => 'Statistics retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Error retrieving statistics: ' . $e->getMessage()
            ], 500);
        }
    }
}