<?php

namespace App\Http\Controllers;

use App\Models\CabinStock;
use App\Models\Item;
use App\Models\Moneys;
use App\Models\TransferReport;
use App\Models\User;
use App\Models\Cabinet;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use function Symfony\Component\Clock\now;

class CabinStockController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        try {
            // Get all cabin stocks with related data
            $cabinStocks = CabinStock::with(['item.type' => function ($query) {
                $query->select('id', 'name','picture'); // only these columns from types
            }, 'user', 'money', 'cobin'])->where('qty','>',0)
                ->orderBy('created_at', 'desc')
                ->get();
            
            return response()->json([
                'success' => true,
                'data' => $cabinStocks
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve cabin stocks: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    
public function transfer(Request $request)
{
    // Validate the request
    $validator = Validator::make($request->all(), [
        'id'       => 'required|exists:cabin_stocks,id',
        'from'     => 'required|exists:cabinets,id',
        'qty'      => 'required|integer|min:1',
        'to'       => 'required|exists:cabinets,id',
        'user_id'  => 'required|exists:users,id',
        'notes'    => 'nullable|string|max:500',
    ]);

    if ($validator->fails()) {
        return response()->json([
            'success' => false,
            'message' => 'Validation failed',
            'errors' => $validator->errors()
        ], 422);
    }

    $validatedData = $validator->validated();

    DB::beginTransaction();

    try {
        // Find source cabin stock
        $sourceCabinStock = CabinStock::with(['item.type:id,name,picture', 'user', 'money', 'cobin'])
            ->where('id', $validatedData['id'])
            ->where('cobin_id', $validatedData['from'])
            ->firstOrFail();

        // Check qty
        if ($sourceCabinStock->qty < $validatedData['qty']) {
            return response()->json([
                'success' => false,
                'message' => 'Insufficient quantity available for transfer'
            ], 400);
        }

        // Deduct source qty
        $sourceCabinStock->decrement('qty', $validatedData['qty']);

        // Find or create destination stock
        $destinationCabinStock = CabinStock::where('item_id', $sourceCabinStock->item_id)
            ->where('cobin_id', $validatedData['to'])
            ->where('purchase_price', $sourceCabinStock->purchase_price)
            ->where('money_id', $sourceCabinStock->money_id)
            ->first();

        if ($destinationCabinStock) {
            $destinationCabinStock->increment('qty', $validatedData['qty']);
        } else {
            $destinationCabinStock = CabinStock::create([
                'item_id'        => $sourceCabinStock->item_id,
                'cobin_id'       => $validatedData['to'],
                'qty'            => $validatedData['qty'],
                'weight'         => $sourceCabinStock->weight,
                'dateInsert'     => $sourceCabinStock->dateInsert,
                'rate'           => $sourceCabinStock->rate,
                'user_id'        => $sourceCabinStock->user_id,
                'purchase_price' => $sourceCabinStock->purchase_price,
                'expiry_date'    => $sourceCabinStock->expiry_date,
                'description'    => $sourceCabinStock->description,
                'money_id'       => $sourceCabinStock->money_id
            ]);
        }

        // Save transfer report
        TransferReport::create([
            'item_id'       => $sourceCabinStock->item_id,
            'from_cabin_id' => $sourceCabinStock->cobin_id,
            'to_cabin_id'   => $destinationCabinStock->cobin_id,
            'quantity'      => $validatedData['qty'],
            'status'        => 'completed',
            'initiated_by'  => $validatedData['user_id'],
            'notes'         => $validatedData['notes'] ?? null,
            'transfer_date' => now(),
            'completed_at'  => now(),
        ]);

        DB::commit();
        $sourceCabinStock->load(['item.type' => function ($query) {
                $query->select('id', 'name'); // only these columns from types
            }, 'user', 'money', 'cobin','picture']);
            
        $destinationCabinStock->load(['item.type' => function ($query) {
                $query->select('id', 'name'); // only these columns from types
            }, 'user', 'money', 'cobin','picture']);
        return response()->json([
            'success' => true,
            'message' => 'Stock transferred successfully',
            'data' => [
                'fromStock' => $sourceCabinStock->fresh(),
                'toStock'   => $destinationCabinStock->fresh(),
            ]
        ], 200);

    } catch (\Exception $e) {
        DB::rollBack();

        return response()->json([
            'success' => false,
            'message' => 'Failed to transfer stock: ' . $e->getMessage()
        ], 500);
    }
}


    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        try {
            // Find the cabin stock with related data
            $cabinStock = CabinStock::with(['item.type' => function ($query) {
                $query->select('id', 'name'); // only these columns from types
            }, 'user', 'money', 'cobin','picture'])->find($id);
            
            if (!$cabinStock) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cabin stock not found'
                ], 404);
            }
            
            return response()->json([
                'success' => true,
                'data' => $cabinStock
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve cabin stock: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        // Find the cabin stock
        $cabinStock = CabinStock::find($id);
        
        if (!$cabinStock) {
            return response()->json([
                'success' => false,
                'message' => 'Cabin stock not found'
            ], 404);
        }
        
        // Validate the request
        $validator = Validator::make($request->all(), [
            'item_id' => 'sometimes|required|exists:items,id',
            'cobin_id' => 'sometimes|required|exists:cabins,id',
            'qty' => 'sometimes|required|integer|min:0',
            'weight' => 'nullable|numeric|min:0',
            'dateInsert' => 'nullable|date',
            'rate' => 'nullable|integer|min:0',
            'user_id' => 'sometimes|required|exists:users,id',
            'purchase_price' => 'nullable|integer|min:0',
            'expiry_date' => 'nullable|date',
            'description' => 'nullable|string|max:500',
            'money_id' => 'sometimes|required|exists:moneys,id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        
        try {
            // Update the cabin stock
            $cabinStock->update($request->all());
            
            DB::commit();
            
            // Refresh the model with relationships
            $cabinStock->load(['item', 'user', 'money', 'cobin']);
            
            return response()->json([
                'success' => true,
                'message' => 'Cabin stock updated successfully',
                'data' => $cabinStock
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to update cabin stock: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        // Find the cabin stock
        $cabinStock = CabinStock::find($id);
        
        if (!$cabinStock) {
            return response()->json([
                'success' => false,
                'message' => 'Cabin stock not found'
            ], 404);
        }

        DB::beginTransaction();
        
        try {
            // Delete the cabin stock
            $cabinStock->delete();
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Cabin stock deleted successfully'
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete cabin stock: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get related data for forms (items, users, etc.)
     */
    public function getFormData()
    {
        try {
            $items = Item::select('id', 'name')->get();
            $users = User::select('id', 'name')->get();
            $moneys = Moneys::select('id', 'name', 'symbol')->get();
            $cabins = Cabinet::select('id', 'name', 'location')->get();
            
            return response()->json([
                'success' => true,
                'data' => [
                    'items' => $items,
                    'users' => $users,
                    'moneys' => $moneys,
                    'cabins' => $cabins
                ]
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve form data: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Search cabin stocks with filters
     */
    public function search(Request $request)
    {
        try {
            $query = CabinStock::with(['item.type' => function ($query) {
                $query->select('id', 'name'); // only these columns from types
            }, 'user', 'money', 'cobin','picture']);
            
            // Filter by item
            if ($request->has('item_id') && $request->item_id) {
                $query->where('item_id', $request->item_id);
            }
            
            // Filter by cabin
            if ($request->has('cobin_id') && $request->cobin_id) {
                $query->where('cobin_id', $request->cobin_id);
            }
            
            // Filter by expiry date range
            if ($request->has('expiry_start') && $request->expiry_start) {
                $query->where('expiry_date', '>=', $request->expiry_start);
            }
            
            if ($request->has('expiry_end') && $request->expiry_end) {
                $query->where('expiry_date', '<=', $request->expiry_end);
            }
            
            // Filter by quantity range
            if ($request->has('min_qty') && $request->min_qty) {
                $query->where('qty', '>=', $request->min_qty);
            }
            
            if ($request->has('max_qty') && $request->max_qty) {
                $query->where('qty', '<=', $request->max_qty);
            }
            
            // Order by
            $orderBy = $request->has('order_by') ? $request->order_by : 'created_at';
            $orderDirection = $request->has('order_direction') ? $request->order_direction : 'desc';
            $query->orderBy($orderBy, $orderDirection);
            
            $cabinStocks = $query->get();
            
            return response()->json([
                'success' => true,
                'data' => $cabinStocks
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Search failed: ' . $e->getMessage()
            ], 500);
        }
    }
}